<?php
// send_payment_reminders.php

// Define Joomla CLI application
define('_JEXEC', 1);

// Set the base path
define('JPATH_BASE', dirname(__DIR__));

// Error logging setup (do this early)
ini_set('log_errors', 1);
ini_set('error_log', JPATH_BASE . '/cli/cron_error.log');

// Check if essential files exist before including
if (!file_exists(JPATH_BASE . '/includes/defines.php')) {
    error_log('Error: defines.php not found in ' . JPATH_BASE . '/includes/');
    echo "Error: Joomla defines.php not found\n";
    exit(1);
}

if (!file_exists(JPATH_BASE . '/includes/framework.php')) {
    error_log('Error: framework.php not found in ' . JPATH_BASE . '/includes/');
    echo "Error: Joomla framework.php not found\n";
    exit(1);
}

// Load Joomla framework
require_once JPATH_BASE . '/includes/defines.php';
require_once JPATH_BASE . '/includes/framework.php';

// Import necessary Joomla classes
jimport('joomla.application.cli');
jimport('joomla.factory');

use Joomla\CMS\Application\CliApplication;
use Joomla\CMS\Factory;

// Define a custom CLI class with all required abstract methods
class PaymentReminderCli extends CliApplication
{
    public function doExecute()
    {
        // Required by CliApplication
    }
    
    public function getName()
    {
        // Required by CMSApplicationInterface
        return 'payment_reminder';
    }
}

try {
    // Bootstrap Joomla application
    JLoader::setup();
    
    // Create and register our CLI application instance
    $app = new PaymentReminderCli();
    Factory::$application = $app; // Register the application with Factory
    
    // Verify application is set
    if (!Factory::getApplication()) {
        throw new Exception('Failed to initialize application in Factory');
    }
    
    // Get current date and tomorrow's date
    $currentDate = new DateTime();
    $tomorrowDate = new DateTime();
    $tomorrowDate->modify('+1 day');
    $tomorrowDateStr = $tomorrowDate->format('Y-m-d');
    
    error_log("Checking payments for: " . $tomorrowDateStr);
    
    // Get database connection
    $db = Factory::getDbo();
    if (!$db) {
        throw new Exception('Failed to get database connection');
    }
    
    $query = $db->getQuery(true);
    
    // Query RSForms submissions
    $query->select($db->quoteName(array('SubmissionId', 'FieldName', 'FieldValue')))
          ->from($db->quoteName('#__rsform_submission_values'))
          ->where($db->quoteName('FormId') . ' = 8')
          ->where($db->quoteName('FieldName') . ' IN (' . 
                  $db->quote('FullName') . ',' . 
                  $db->quote('Date_of_Payment') . ',' . 
                  $db->quote('Amount_Paid_to_Maid') . ')');
    
    $db->setQuery($query);
    $results = $db->loadObjectList();
    
    if ($results === null) {
        throw new Exception('Database query failed: ' . $db->getErrorMsg());
    }
    
    // Process results into a structured array
    $submissions = array();
    foreach ($results as $row) {
        if (!isset($submissions[$row->SubmissionId])) {
            $submissions[$row->SubmissionId] = array();
        }
        $submissions[$row->SubmissionId][$row->FieldName] = $row->FieldValue;
    }
    
    // Filter submissions for tomorrow's payments
    $duePayments = array();
    foreach ($submissions as $submission) {
        if (isset($submission['Date_of_Payment']) && 
            date('Y-m-d', strtotime($submission['Date_of_Payment'])) === $tomorrowDateStr) {
            $duePayments[] = $submission;
        }
    }
    
    error_log("Found " . count($duePayments) . " payments due tomorrow");
    
    // If there are payments due, send email
    if (!empty($duePayments)) {
        // Build email content
        $emailBody = "Maids Payments Due Tomorrow (" . $tomorrowDate->format('Y-m-d') . "):\n\n";
        $emailBody .= sprintf("%-30s %-20s\n", "Name", "Amount Paid");
        $emailBody .= str_repeat("-", 50) . "\n";
        
        foreach ($duePayments as $payment) {
            $emailBody .= sprintf(
                "%-30s %-20s\n",
                $payment['FullName'] ?? 'Unknown',
                $payment['Amount_Paid_to_Maid'] ?? '0'
            );
        }
        
        // Setup JMail
        $mailer = Factory::getMailer();
        if (!$mailer) {
            throw new Exception('Failed to initialize mailer');
        }
        
        $config = Factory::getConfig();
        if (!$config) {
            throw new Exception('Failed to get configuration');
        }
        
        $sender = array(
            'admin@lafabsolution.com',
            'Lafab Solutions Maids System'
        );
        
        error_log("Attempting to send email...");
        
        $mailer->setSender($sender);
        $mailer->addRecipient(array(
            'admin@lafabsolution.com',
            'mubmart7@gmail.com'
        ));
        $mailer->setSubject('Maids Payments Due Tomorrow');
        $mailer->setBody($emailBody);
        
        // Send the email
        $send = $mailer->Send();
        if ($send !== true) {
            throw new Exception('Email sending failed: ' . (is_object($send) ? $send->getMessage() : print_r($send, true)));
        }
        
        error_log("Email sent successfully");
        echo "Payment reminders sent successfully\n";
    } else {
        echo "No payments due tomorrow\n";
    }
    
} catch (Exception $e) {
    error_log('Cron job error: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
    echo "Error occurred: " . $e->getMessage() . "\n";
    exit(1);
}

exit(0);